package pt.unl.fct.di.novasys.babel.protocols.antientropy;

import java.io.IOException;
import java.net.InetAddress;
import java.util.HashSet;
import java.util.Properties;
import java.util.Random;
import java.util.Set;
import java.util.TreeSet;
import java.util.UUID;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.common.hash.BloomFilter;
import com.google.common.hash.Funnels;

import pt.unl.fct.di.novasys.babel.core.GenericProtocol;
import pt.unl.fct.di.novasys.babel.exceptions.HandlerRegistrationException;
import pt.unl.fct.di.novasys.babel.generic.ProtoMessage;
import pt.unl.fct.di.novasys.babel.protocols.antientropy.messages.AntiEntrophyAnnounce;
import pt.unl.fct.di.novasys.babel.protocols.antientropy.timers.AntiEntrophyTimer;
import pt.unl.fct.di.novasys.babel.protocols.secure.dissemination.messages.SignedIdentifiableProtoMessage;
import pt.unl.fct.di.novasys.babel.protocols.secure.dissemination.notifications.IdentifiableMessageNotification;
import pt.unl.fct.di.novasys.babel.protocols.secure.dissemination.requests.MissingIdentifiableMessageRequest;
import pt.unl.fct.di.novasys.babel.protocols.secure.membership.Peer;
import pt.unl.fct.di.novasys.babel.protocols.secure.membership.notifications.NeighborDown;
import pt.unl.fct.di.novasys.babel.protocols.secure.membership.notifications.NeighborUp;
import pt.unl.fct.di.novasys.babel.protocols.general.notifications.ChannelAvailableNotification;
import pt.unl.fct.di.novasys.channel.secure.auth.AuthChannel;
import pt.unl.fct.di.novasys.channel.secure.events.SecureInConnectionDown;
import pt.unl.fct.di.novasys.channel.secure.events.SecureInConnectionUp;
import pt.unl.fct.di.novasys.channel.secure.events.SecureOutConnectionDown;
import pt.unl.fct.di.novasys.channel.secure.events.SecureOutConnectionFailed;
import pt.unl.fct.di.novasys.channel.secure.events.SecureOutConnectionUp;
import pt.unl.fct.di.novasys.network.data.Host;
import pt.unl.fct.di.novasys.babel.metrics.Counter;
import pt.unl.fct.di.novasys.babel.metrics.Metric;

public class AntiEntropy extends GenericProtocol {

	private class Element implements Comparable<Element> {

		private final SignedIdentifiableProtoMessage m;

		private final Long timestamp;

		private final short protocolSource;

		public Element(SignedIdentifiableProtoMessage m, short protocolSource) {
			this.m = m;
			this.timestamp = System.currentTimeMillis();
			this.protocolSource = protocolSource;
		}

		public UUID getMID() {
			return this.m.getMID();
		}

		public SignedIdentifiableProtoMessage getMessage() {
			return this.m;
		}

		public long getTimestamp() {
			return this.timestamp;
		}

		public short getProtocolSource() {
			return this.protocolSource;
		}

		@Override
		public int compareTo(Element e) {
			return this.timestamp.compareTo(e.timestamp);
		}

	}

	private static final Logger logger = LogManager.getLogger(AntiEntropy.class);

	public final static short PROTOCOL_ID = 1900;
	public final static String PROTOCOL_NAME = "AntiEntrophy";

	public final static String PAR_CHANNEL_ADDRESS = "AntiEntrophy.Channel.Address";
	public final static String PAR_CHANNEL_PORT = "AntiEntrophy.Channel.Port";

	public final static String PAR_PERIOD = "AntiEntrophy.Period";
	public final static long DEFAULT_PERIOD = 60000;
	public final long antientropyPeriod;

	public final static String PAR_REMOVE_TIME = "AntiEntrophy.GCTimeout";
	public final static long DEFAULT_REMOVE_TIME = 600000;
	public final long removeTimeWindow;

	public final static String PAR_BLOOM_FILTER_FPP = "AntiEntophy.BloomFilter.FPP";
	public final static double DEFAULT_BLOOM_FILTER_FPP = 0.0001;
	public final double bloomFilterFPP;

	// The grace period represents a time after registering a message in which a
	// node will
	// not send that message to a peer even if that peer does not report missing
	// that message.
	// Symmetrically when a message has less time to be garbaged collected than this
	// grace
	// period a node will also not send it to a peer even if it reports not having
	// it (to
	// avoid infinite cycles in the network.
	public final static String PAR_GRACE_PERIOD = "AntiEntrophy.GracePeriod";
	public final static long DEFAULT_GRACE_PERIOD = 90000;
	public final long gracePeriod;

	public int networkPort;

	protected int channelId;
	protected Peer myself;

	private Set<Peer> pending;
	private Set<Peer> connectedNeighbors;

	private TreeSet<Element> buffer;

	private final boolean managingChannel;
	private final Random r;

	private final Counter sentMessagesCounter;


	public AntiEntropy(Properties properties, Host myAddr) throws IOException, HandlerRegistrationException {
		super(PROTOCOL_NAME, PROTOCOL_ID);
		this.myself = new Peer(babelSecurity.getDefaultIdentity().identity(), myAddr);

		this.pending = new TreeSet<>();
		this.connectedNeighbors = new TreeSet<>();

		this.buffer = new TreeSet<Element>();

		if (properties.containsKey(PAR_PERIOD))
			this.antientropyPeriod = Long.parseLong(properties.getProperty(PAR_PERIOD));
		else
			this.antientropyPeriod = DEFAULT_PERIOD;

		if (properties.containsKey(PAR_REMOVE_TIME))
			this.removeTimeWindow = Long.parseLong(properties.getProperty(PAR_REMOVE_TIME));
		else
			this.removeTimeWindow = DEFAULT_REMOVE_TIME;

		if (properties.containsKey(PAR_GRACE_PERIOD))
			this.gracePeriod = Long.parseLong(properties.getProperty(PAR_GRACE_PERIOD));
		else
			this.gracePeriod = DEFAULT_GRACE_PERIOD;

		if (properties.containsKey(PAR_BLOOM_FILTER_FPP))
			this.bloomFilterFPP = Double.parseDouble(properties.getProperty(PAR_BLOOM_FILTER_FPP));
		else
			this.bloomFilterFPP = DEFAULT_BLOOM_FILTER_FPP;

		this.r = new Random(System.currentTimeMillis());

		//Metrics initialization
		this.sentMessagesCounter = registerMetric(new Counter.Builder("SentMessages", Metric.Unit.NONE).build());

		String address = null;
		String port = null;

		if (properties.containsKey(PAR_CHANNEL_ADDRESS) && properties.containsKey(PAR_CHANNEL_PORT)) {
			address = properties.getProperty(PAR_CHANNEL_ADDRESS);
			port = properties.getProperty(PAR_CHANNEL_PORT);
			this.networkPort = Short.parseShort(port);
			if (myAddr == null)
				myAddr = new Host(InetAddress.getByName(address), this.networkPort);
		} else if (myAddr != null) {
			address = myAddr.getAddress().getHostAddress();
			this.networkPort = myAddr.getPort();
			port = this.networkPort + "";
		}

		if (address != null && port != null) { // we have information to build our own channel
			this.managingChannel = true;

			Properties channelProps = new Properties();
			channelProps.setProperty(AuthChannel.ADDRESS_KEY, address); // The address to bind to
			channelProps.setProperty(AuthChannel.PORT_KEY, port); // The port to bind to
			this.channelId = createSecureChannel(AuthChannel.NAME, channelProps);
			setDefaultChannel(this.channelId);

			/*---------------------- Register Message Serializers ---------------------- */
			registerMessageSerializer(channelId, AntiEntrophyAnnounce.MSG_CODE, AntiEntrophyAnnounce.serializer);

			/*---------------------- Register Message Handlers ---------------------- */
			registerMessageHandler(channelId, AntiEntrophyAnnounce.MSG_CODE, this::uponAntiEntrophyAnnounceMessage, this::uponMessageFailed);

			/*-------------------- Register Notification Handlers ------------------------ */
			subscribeNotification(NeighborUp.NOTIFICATION_ID, this::uponNeighborUp);
			subscribeNotification(NeighborDown.NOTIFICATION_ID, this::uponNeighborDown);

			/*-------------------- Register Channel Event ------------------------------- */
			registerChannelEventHandler(channelId, SecureOutConnectionDown.EVENT_ID, this::uponOutConnectionDown);
			registerChannelEventHandler(channelId, SecureOutConnectionFailed.EVENT_ID, this::uponOutConnectionFailed);
			registerChannelEventHandler(channelId, SecureOutConnectionUp.EVENT_ID, this::uponOutConnectionUp);
			registerChannelEventHandler(channelId, SecureInConnectionUp.EVENT_ID, this::uponInConnectionUp);
			registerChannelEventHandler(channelId, SecureInConnectionDown.EVENT_ID, this::uponInConnectionDown);
		} else { // we will not be using our own channel and instead will rely on a shared
					// channel
			this.managingChannel = false;
			subscribeNotification(ChannelAvailableNotification.NOTIFICATION_ID, this::uponChannelAvailableNotifiction);
		}

		subscribeNotification(IdentifiableMessageNotification.NOTIFICATION_ID,
				this::uponIdentifiableMessageNotification);

		/*-------------------- Register Notification Handlers ------------------------ */
		subscribeNotification(NeighborUp.NOTIFICATION_ID, this::uponNeighborUp);
		subscribeNotification(NeighborDown.NOTIFICATION_ID, this::uponNeighborDown);

		/*-------------------- Register Timer Handlers ------------------------ */
		registerTimerHandler(AntiEntrophyTimer.PROTO_ID, this::uponAntiEntrophyTimer);

	}

	@Override
	public void init(Properties props) throws HandlerRegistrationException, IOException {
		setupPeriodicTimer(new AntiEntrophyTimer(), this.antientropyPeriod, this.antientropyPeriod);
	}

    private void uponMessageFailed(ProtoMessage msg, Host to, byte[] toId, short destProto, Throwable cause, int channelId) {
        logger.warn("Message failed: msg={}, to={}, destProto={}, cause={}, channelId={}",
                msg, new Peer(toId, to), destProto, cause, channelId);
    }

	private void uponAntiEntrophyTimer(AntiEntrophyTimer timer, long time) {
		if (this.connectedNeighbors.size() > 0) {
			Peer peer = this.connectedNeighbors.toArray(new Peer[this.connectedNeighbors.size()])
					[this.r.nextInt(this.connectedNeighbors.size())];

			BloomFilter<String> bf = BloomFilter.create(Funnels.unencodedCharsFunnel(), this.buffer.size(),
					this.bloomFilterFPP);

			for (Element e : this.buffer) {
				bf.put(e.getMID().toString());
			}

			logger.debug("Sent an AntiEntrophyAnnounce to " + peer + " with " + buffer.size() + " elements.");
			sendMessage(new AntiEntrophyAnnounce(myself, bf, PROTOCOL_ID), peer);
			sentMessagesCounter.inc();
		}
		// clean up stage
		Set<Element> gc = new HashSet<AntiEntropy.Element>();

		long cleanUpBarrier = System.currentTimeMillis() - this.removeTimeWindow;

		for (Element e : this.buffer) {
			if (e.getTimestamp() < cleanUpBarrier)
				gc.add(e);
			else
				break;
		}

		this.buffer.removeAll(gc);
		logger.debug("Purged " + gc.size() + " elements from my buffer.");
	}

	/*
	 * ----------------------- Message handlers--------------------
	 */

	private void uponAntiEntrophyAnnounceMessage(AntiEntrophyAnnounce msg, Host sender, byte[] senderId, short protoID, int cID) {
        Peer peer = new Peer(senderId, sender);
		logger.debug("Received and AntiEntrophyAnnounce reporting " + msg.getSetSize() + " messages from " + sender);

		long now = System.currentTimeMillis();

		for (Element e : this.buffer) {
			// Following condition verifies if the element in the buffer is outside the
			// grace
			// period protection
			if (e.getTimestamp() < (now - gracePeriod) && e.getTimestamp() > (now - removeTimeWindow + gracePeriod)) {
				if (!msg.contains(e.getMID().toString())) {
					logger.info("Requesting recovery of message " + e.getMID() + " to " + sender);
					sendRequest(new MissingIdentifiableMessageRequest(e.getMessage(), peer), e.getProtocolSource());
				} else {
					logger.debug("Messagee " + e.getMID() + " appears to be known by " + sender);
				}
			} else {
				logger.trace("Message " + e.getMID() + " excluded due to grace period (" + gracePeriod
						+ "). Received at: " + e.getTimestamp() + " GC: " + this.removeTimeWindow + " now: " + now);
			}
		}
	}

	/*
	 * ------------------- Notification handlers -------------------
	 */

	private void uponChannelAvailableNotifiction(ChannelAvailableNotification event, short protoID) {
		if (myself.getHost() == null) { // we will only do this once for the first notification of this type (we could
								// put more filters)
			myself.setHost(event.getChannelListenData());
			this.networkPort = myself.getPort();

			this.channelId = event.getChannelID();
			registerSharedChannel(this.channelId);

			/*---------------------- Register Message Serializers ---------------------- */
			registerMessageSerializer(channelId, AntiEntrophyAnnounce.MSG_CODE, AntiEntrophyAnnounce.serializer);

			try {
				/*---------------------- Register Message Handlers ---------------------- */
				registerMessageHandler(channelId, AntiEntrophyAnnounce.MSG_CODE, this::uponAntiEntrophyAnnounceMessage);

				/*-------------------- Register Channel Event ------------------------------- */
				registerChannelEventHandler(channelId, SecureOutConnectionDown.EVENT_ID, this::uponOutConnectionDown);
				registerChannelEventHandler(channelId, SecureOutConnectionFailed.EVENT_ID, this::uponOutConnectionFailed);
				registerChannelEventHandler(channelId, SecureOutConnectionUp.EVENT_ID, this::uponOutConnectionUp);
				registerChannelEventHandler(channelId, SecureInConnectionUp.EVENT_ID, this::uponInConnectionUp);
				registerChannelEventHandler(channelId, SecureInConnectionDown.EVENT_ID, this::uponInConnectionDown);
			} catch (HandlerRegistrationException e) {
				e.printStackTrace();
				System.exit(1);
			}
		}
	}

	private void uponIdentifiableMessageNotification(IdentifiableMessageNotification event, short protoID) {
		this.buffer.add(new Element(event.getMessage(), event.getSourceProtocol()));
	}

	private void uponNeighborUp(NeighborUp up, short protoID) {
		// Transformation of Host to have network port
		Peer peer = up.getPeer();

		if (managingChannel) {
			if (!this.connectedNeighbors.contains(peer) && this.pending.add(peer))
				openConnection(peer);
		} else {
			this.connectedNeighbors.add(peer);
		}
	}

	private void uponNeighborDown(NeighborDown down, short protoID) {
		// Transformation of Host to have network port
		Peer peer = down.getPeer().clone();

		if (managingChannel) {
			if (this.connectedNeighbors.remove(peer)) {
				closeConnection(peer);
			} else if (this.pending.remove(peer)) {
				closeConnection(peer);
			}
		} else {
			this.connectedNeighbors.remove(peer);
		}
	}

	/*
	 * --------------------------------- Channel Events ----------------------------
	 */

	private void uponOutConnectionDown(SecureOutConnectionDown event, int channelId) {
		Peer peer = new Peer(event.getNodeId(), event.getNode());
		logger.trace("Peer {} is down, cause: {}", peer, event.getCause());

		if (this.connectedNeighbors.contains(peer) || this.pending.contains(peer)) {
			this.connectedNeighbors.remove(peer);
			this.pending.add(peer);
			openConnection(peer);
		}
	}

	private void uponOutConnectionFailed(SecureOutConnectionFailed<?> event, int channelId) {
		Peer peer = new Peer(event.getNodeId(), event.getNode());
		logger.trace("Connection to peer {} failed, cause: {}", peer, event.getCause());

		if (this.connectedNeighbors.contains(peer) || this.pending.contains(peer)) {
			this.connectedNeighbors.remove(peer);
			this.pending.add(peer);
			openConnection(peer);
		}
	}

	private void uponOutConnectionUp(SecureOutConnectionUp event, int channelId) {
		Peer peer = new Peer(event.getNodeId(), event.getNode());
		logger.trace("Peer (out) {} is up", peer);

		if (!this.connectedNeighbors.contains(peer) && !this.pending.contains(peer)) {
			closeConnection(peer);
			return;
		}

		this.pending.remove(peer);
		this.connectedNeighbors.add(peer);
	}

	private void uponInConnectionUp(SecureInConnectionUp event, int channelId) {
		logger.trace("Peer (in) {} is up", new Peer(event.getNodeId(), event.getNode()));
	}

	private void uponInConnectionDown(SecureInConnectionDown event, int channelId) {
		logger.trace("Connection from peer {} is down, cause: {}",
				new Peer(event.getNodeId(), event.getNode()), event.getCause());
	}

	public Peer getPeer() {
		return this.myself;
	}

	public Host getHost() {
		return this.myself.getHost();
	}

    private void openConnection(Peer peer) {
        super.openConnection(peer.getHost(), peer.getIdentity());
    }

    private void closeConnection(Peer peer) {
        super.closeConnection(peer.getIdentity());
    }

    private void sendMessage(ProtoMessage msg, Peer dest) {
        super.sendMessage(msg, dest.getIdentity());
    }

}
