package pt.unl.fct.di.novasys.babel.protocols.eagerpush;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Properties;
import java.util.Random;
import java.util.Set;
import java.util.TreeSet;
import java.util.UUID;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import pt.unl.fct.di.novasys.babel.core.GenericProtocol;
import pt.unl.fct.di.novasys.babel.exceptions.HandlerRegistrationException;
import pt.unl.fct.di.novasys.babel.protocols.dissemination.notifications.BroadcastDelivery;
import pt.unl.fct.di.novasys.babel.protocols.dissemination.notifications.IdentifiableMessageNotification;
import pt.unl.fct.di.novasys.babel.protocols.dissemination.requests.BroadcastRequest;
import pt.unl.fct.di.novasys.babel.protocols.dissemination.requests.MissingIdentifiableMessageRequest;
import pt.unl.fct.di.novasys.babel.protocols.eagerpush.messages.GossipMessage;
import pt.unl.fct.di.novasys.babel.protocols.general.notifications.ChannelAvailableNotification;
import pt.unl.fct.di.novasys.babel.protocols.membership.notifications.NeighborDown;
import pt.unl.fct.di.novasys.babel.protocols.membership.notifications.NeighborUp;
import pt.unl.fct.di.novasys.channel.tcp.TCPChannel;
import pt.unl.fct.di.novasys.channel.tcp.events.InConnectionDown;
import pt.unl.fct.di.novasys.channel.tcp.events.InConnectionUp;
import pt.unl.fct.di.novasys.channel.tcp.events.OutConnectionDown;
import pt.unl.fct.di.novasys.channel.tcp.events.OutConnectionFailed;
import pt.unl.fct.di.novasys.channel.tcp.events.OutConnectionUp;
import pt.unl.fct.di.novasys.network.data.Host;

public class EagerPushGossipBroadcast extends GenericProtocol {

    private static final Logger logger = LogManager.getLogger(EagerPushGossipBroadcast.class);

    public final static short PROTOCOL_ID = 600;
    public final static String PROTOCOL_NAME = "EagerPushGossipBroadcast";
    
    public final static String PAR_CHANNEL_ADDRESS = "EagerPushGossipBroadcast.Channel.Address";
    public final static String PAR_CHANNEL_PORT = "EagerPushGossipBroadcast.Channel.Port";
    
    public final static String PAR_FANOUT = "EagerPushGossipBroadcast.Fanout";
    public final static String DEFAULT_FANOUT = "4";
    
    public final static String PAR_DELIVERY_TIMEOUT = "EagerPushGossipBroadcast.DeliveredTimeout";
    public final static String DEFAULT_DELIVERY_TIEMEOUT = "600000";
    
    public final static String PAR_SUPPORT_ANTIENTROPHY = "EagerPushGosssipBroadcast.SupportAntiEntropy";
    public final static boolean DEFAULT_SUPPORT_ANTIENTROPHY = false;
    
    public final int fanout;
    public final long removeTimeWindow;
    
    public final int networkPort;
    
    public final boolean supportAntiEntrophy;
    
    protected int channelId;
    protected final Host myself;
    
    private Set<Host> pending;
    private Set<Host> connectedNeighbors;
    private Set<Host> neighbors;
    
    private LinkedList<UUID> receivedInOrder;
    private HashMap<UUID,Long> rerceiveTimestamps;
    
    private final Random r;
    
	public EagerPushGossipBroadcast(String channelName, Properties properties, Host myself) throws IOException, HandlerRegistrationException {
		super(PROTOCOL_NAME, PROTOCOL_ID);
		this.myself = myself;
		
		this.pending = new TreeSet<Host>();
		this.connectedNeighbors = new TreeSet<Host>();
		this.neighbors = new TreeSet<Host>();
		
		this.receivedInOrder = new LinkedList<UUID>();
		this.rerceiveTimestamps = new HashMap<UUID, Long>();
		
		this.fanout = Integer.parseInt(properties.getProperty(EagerPushGossipBroadcast.PAR_FANOUT, EagerPushGossipBroadcast.DEFAULT_FANOUT));
		this.removeTimeWindow = Long.parseLong(properties.getProperty(EagerPushGossipBroadcast.PAR_DELIVERY_TIMEOUT, EagerPushGossipBroadcast.DEFAULT_DELIVERY_TIEMEOUT));
		
		this.r = new Random(System.currentTimeMillis());
		
		if(properties.containsKey(PAR_SUPPORT_ANTIENTROPHY)) {
			this.supportAntiEntrophy = Boolean.parseBoolean(properties.getProperty(PAR_SUPPORT_ANTIENTROPHY));
		} else {
			this.supportAntiEntrophy = DEFAULT_SUPPORT_ANTIENTROPHY;
		}
		
		String address = null;
        String port = null; 
        
        if(properties.containsKey(EagerPushGossipBroadcast.PAR_CHANNEL_ADDRESS))
        	address = properties.getProperty(EagerPushGossipBroadcast.PAR_CHANNEL_ADDRESS);
        else
        	address = myself.getAddress().getHostAddress().toString();
        
        if(properties.containsKey(EagerPushGossipBroadcast.PAR_CHANNEL_PORT)) {
        	port = properties.getProperty(EagerPushGossipBroadcast.PAR_CHANNEL_PORT);
        	this.networkPort = Integer.parseInt(port);
        } else {
        	this.networkPort = myself.getPort();
        	port = this.networkPort + "";
        }
        
        Properties channelProps = new Properties();
        channelProps.setProperty(TCPChannel.ADDRESS_KEY, address); //The address to bind to
        channelProps.setProperty(TCPChannel.PORT_KEY, port); //The port to bind to
        this.channelId = createChannel(TCPChannel.NAME, channelProps);

        setDefaultChannel(channelId);
        
        /*---------------------- Register Message Serializers ---------------------- */
        registerMessageSerializer(channelId, GossipMessage.MSG_CODE, GossipMessage.serializer);
        
        /*---------------------- Register Message Handlers ---------------------- */
        registerMessageHandler(channelId, GossipMessage.MSG_CODE, this::uponGossipMessage);
        
        /*---------------------- Register Requests Handlers -------------------------- */
        registerRequestHandler(BroadcastRequest.REQUEST_ID, this::uponBroadcastRequest);
        
        /*-------------------- Register Notification Handlers ------------------------ */
        subscribeNotification(NeighborUp.NOTIFICATION_ID, this::uponNeighborUp);
        subscribeNotification(NeighborDown.NOTIFICATION_ID, this::uponNeighborDown);
	
        if(this.supportAntiEntrophy)
        	registerRequestHandler(MissingIdentifiableMessageRequest.REQUEST_ID, this::uponMissingMessageRequest);
        
        /*-------------------- Register Channel Event ------------------------------- */
        registerChannelEventHandler(channelId, OutConnectionDown.EVENT_ID, this::uponOutConnectionDown);
        registerChannelEventHandler(channelId, OutConnectionFailed.EVENT_ID, this::uponOutConnectionFailed);
        registerChannelEventHandler(channelId, OutConnectionUp.EVENT_ID, this::uponOutConnectionUp);
        registerChannelEventHandler(channelId, InConnectionUp.EVENT_ID, this::uponInConnectionUp);
        registerChannelEventHandler(channelId, InConnectionDown.EVENT_ID, this::uponInConnectionDown);
	}

	@Override
	public void init(Properties props) throws HandlerRegistrationException, IOException {
		//emit the notification to indicate that the channel owned by this protocol can 
		//be used by other protocols
		triggerNotification(new ChannelAvailableNotification(PROTOCOL_ID, PROTOCOL_NAME, this.channelId, TCPChannel.NAME, myself));

	}
	
	/* --------------------------------- Request handlers ---------------------------- */
	
	private void uponBroadcastRequest(BroadcastRequest request, short protoID) {
		GossipMessage msg = new GossipMessage(request.getTimestamp(), myself, request.getPayload(), protoID);
		deliverMessage(msg.clone());
		
		ArrayList<Host> validTargets = new ArrayList<Host>(this.connectedNeighbors);
		
		int toSend = this.fanout;
		while(validTargets.size() > 0 && toSend > 0) {
			sendMessage(msg, validTargets.remove(r.nextInt(validTargets.size())));
		}
		
		cleanUp();
	}
	
	private void deliverMessage(GossipMessage msg) {
		this.receivedInOrder.addLast(msg.getMID());
		this.rerceiveTimestamps.put(msg.getMID(), System.currentTimeMillis());
		
		triggerNotification(new BroadcastDelivery(msg.getSender(), msg.getPayload(), msg.getTimestamp()));
		if(this.supportAntiEntrophy)
			triggerNotification(new IdentifiableMessageNotification(msg, EagerPushGossipBroadcast.PROTOCOL_ID));
	}
	
	private void cleanUp() {
		long now = System.currentTimeMillis();
		while(!this.receivedInOrder.isEmpty() && this.rerceiveTimestamps.get(this.receivedInOrder.pollFirst()) + this.removeTimeWindow < now)
			this.rerceiveTimestamps.remove(this.receivedInOrder.removeFirst());
	}
	
	/* --------------------------------- Message handlers ---------------------------- */
	
	private void uponGossipMessage(GossipMessage msg, Host sender, short protoID, int cID) {
		if(!this.rerceiveTimestamps.containsKey(msg.getMID()))
			deliverMessage(msg.clone());
		
		msg.incrementHopCount();
		
		ArrayList<Host> validTargets = new ArrayList<Host>(this.connectedNeighbors);
		validTargets.remove(sender);
		validTargets.remove(msg.getSender());
		
		int toSend = this.fanout;
		while(validTargets.size() > 0 && toSend > 0) {
			sendMessage(msg, validTargets.remove(r.nextInt(validTargets.size())));
		}
		
		cleanUp();
	}
	
	/* --------------------------------- Notification handlers ---------------------------- */
	
	private void uponNeighborUp(NeighborUp up, short protoID) {
		Host h = new Host(up.getPeer().getAddress(), this.networkPort); //Transformation of Host to have network port of this protocol.
		
		this.neighbors.add(h);
		
		if(!this.connectedNeighbors.contains(h) && this.pending.add(h))
			openConnection(h);
	}
	
	private void uponNeighborDown(NeighborDown down, short protoID) {
		Host h = new Host(down.getPeer().getAddress(), this.networkPort); //Transformation of Host to have network port of this protocol.
		
		this.neighbors.remove(h);
		
		if(this.connectedNeighbors.remove(h)) {
			closeConnection(h);
		} else if(this.pending.remove(h)) {
			closeConnection(h);
		}
	}
	
	private void uponMissingMessageRequest(MissingIdentifiableMessageRequest req, short protoID) {
		logger.debug("Received an anti-entrophy request indicating that " + req.getDestination() + " is missing " + req.getMessage().getMID());

		Host d = null;
		if(this.connectedNeighbors.contains(req.getDestination()) || this.pending.contains(req.getDestination())) {
			d = req.getDestination();
			logger.debug("Destination in requeest is among my connections (active or pending)");
		} else {
			Host tmp = new Host(req.getDestination().getAddress(), this.networkPort);
			if(this.connectedNeighbors.contains(tmp) || this.pending.contains(tmp)) {
				d = tmp;
				logger.debug("Destination was translated to my protocol channel (" + tmp + ") and was found among my connections (active or pending)");
			}
		}
		
		if(d != null) {
			sendMessage(req.getMessage(), d);
			logger.info("Recoved message " + req.getMessage().getMID() + " to " + d);
		} else {
			logger.info("Unable to recover message " + req.getMessage().getMID() + " to " + req.getDestination());
		}
	
	}
    /* --------------------------------- Channel Events ---------------------------- */

    private void uponOutConnectionDown(OutConnectionDown event, int channelId) {
    	Host h = event.getNode();
    	logger.trace("Host {} is down, cause: {}", h, event.getCause());   	
    	
    	if(this.neighbors.contains(h)) {
      		this.connectedNeighbors.remove(h);
      		this.pending.add(h);
      		openConnection(h);
      	} else {
      		this.connectedNeighbors.remove(h);
      		this.pending.remove(h);
      		closeConnection(h);
      	}
    }

    private void uponOutConnectionFailed(OutConnectionFailed<?> event, int channelId) {
    	Host h = event.getNode();
    	logger.trace("Connection to host {} failed, cause: {}", h, event.getCause());
       
    	if(this.neighbors.contains(h)) {
      		this.connectedNeighbors.remove(h);
      		this.pending.add(h);
      		openConnection(h);
      	} else {
      		this.connectedNeighbors.remove(h);
      		this.pending.remove(h);
      		closeConnection(h);
      	}
    }

    private void uponOutConnectionUp(OutConnectionUp event, int channelId) {
    	Host h = event.getNode();
    	logger.trace("Host (out) {} is up", h);
        
    	if(this.neighbors.contains(h)) {
    		this.pending.remove(h);
    		this.connectedNeighbors.add(h);
    	} else {
    		this.pending.remove(h);
    		this.connectedNeighbors.remove(h);
    		closeConnection(h);
      	}
    }

    private void uponInConnectionUp(InConnectionUp event, int channelId) {
        logger.trace("Host (in) {} is up", event.getNode());
    }

    private void uponInConnectionDown(InConnectionDown event, int channelId) {
        logger.trace("Connection from host {} is down, cause: {}", event.getNode(), event.getCause());
    }

}
