package pt.unl.fct.di.tardis.babel.iot.counterprotocol;

import java.io.IOException;
import java.util.Properties;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.github.yafna.raspberry.grovepi.digital.*;

import io.helins.linux.gpio.*;
import pt.unl.fct.di.novasys.babel.core.GenericProtocol;
import pt.unl.fct.di.novasys.babel.exceptions.HandlerRegistrationException;
import pt.unl.fct.di.novasys.babel.protocols.dissemination.requests.BroadcastRequest;
import pt.unl.fct.di.novasys.network.data.Host;
// stolen from disseminationApp
import pt.unl.fct.di.tardis.babel.iot.counterprotocol.messages.UserMessage;

public class IoTCounterProtocol extends GenericProtocol {

    private static final Logger logger =
        LogManager.getLogger(IoTCounterProtocol.class);

    public final static short PROTO_ID = 9000;
    public final static String PROTO_NAME = "IoTCounterProtocol";
    public final static String INNER_SENSOR_LINE = "ctr.inner.line";
    public final static String OUTER_SENSOR_LINE = "ctr.outer.line";
    public final static String INNER_SENSOR_ID = "ctr.inner.id";
    public final static String OUTER_SENSOR_ID = "ctr.outer.id";

    public final static String LED0_LINE = "ctr.led0.line";
    public final static String LED1_LINE = "ctr.led1.line";

    private static final String DEFAULT_PATH_TO_DEVICE = "/dev/gpiochip4";

    private GpioDevice device;
    private GpioEvent event;
    private GpioEventWatcher watcher;
    private GpioHandleRequest request;

    private final Host myself;

    private long thread_id;

    // assuming sensors placed in parallel, at an entry/exit point
    private DigitalInputDevice
        inner_sensor; // the one closest to the inside of the room
    private DigitalInputDevice
        outer_sensor; // the one closest to the outside of the room

    private DigitalOutputDevice led0;
    private DigitalOutputDevice led1;

    public IoTCounterProtocol(Host host) throws HandlerRegistrationException {
        super(IoTCounterProtocol.PROTO_NAME, IoTCounterProtocol.PROTO_ID);

        this.myself = host;

        try {
            this.device = new GpioDevice(DEFAULT_PATH_TO_DEVICE);
        } catch (IOException e) {
            logger.error("Could not initialize the GpioDevice: ", e);
            this.device = null;
        }

        try {
            this.watcher = new GpioEventWatcher();
        } catch (IOException e) {
            logger.error("Could not initialize the GpioEventWatcher: ", e);
            this.watcher = null;
        }

        try {
            this.request =
                new GpioHandleRequest().setConsumer("counter-proto").setFlags(
                    new GpioFlags().setOutput());
        } catch (Exception e) {
            logger.error("Could not initialize the GpioHandleRequest: ", e);
            this.request = null;
        }

        this.thread_id = 0L;
    }

    @Override
    public void init(Properties props)
        throws HandlerRegistrationException, IOException {
        int inner_sensor_line = 0, outer_sensor_line = 0;
        int inner_sensor_id = 0, outer_sensor_id = 0;
        int led0_line = 0, led1_line = 0;

        if (props.containsKey(INNER_SENSOR_LINE)) {
            inner_sensor_line =
                Short.parseShort(props.getProperty(INNER_SENSOR_LINE));
            logger.debug("IoTCounterProtocol is configured to use inner "
                         + "sensor on digital line: " + inner_sensor_line);
        } else {
            logger.error("The applicaiton requires the line number of the"
                         + "inner-facing sensor in use. Parameter: '" +
                         INNER_SENSOR_LINE + "'");
            System.exit(1);
        }
        if (props.containsKey(INNER_SENSOR_ID)) {
            inner_sensor_id =
                Short.parseShort(props.getProperty(INNER_SENSOR_ID));
        } else {
            inner_sensor_id = inner_sensor_line;
            logger.debug(
                "Assuming inner sensor ID same as inner sensor line number.");
        }
        logger.debug(
            "IoTCounterProtocol is configured to use inner sensor ID: " +
            inner_sensor_id);

        if (props.containsKey(OUTER_SENSOR_LINE)) {
            outer_sensor_line =
                Short.parseShort(props.getProperty(OUTER_SENSOR_LINE));
            logger.debug("IoTCounterProtocol is configured to use outer "
                         + "sensor on digital line: " + outer_sensor_line);
        } else {
            logger.error("The applicaiton requires the line number of the"
                         + "outer-facing sensor in use. Parameter: '" +
                         OUTER_SENSOR_LINE + "'");
            System.exit(1);
        }
        if (props.containsKey(OUTER_SENSOR_ID)) {
            outer_sensor_id =
                Short.parseShort(props.getProperty(OUTER_SENSOR_ID));
        } else {
            outer_sensor_id = outer_sensor_line;
            logger.debug(
                "Assuming outer sensor ID same as outer sensor line number.");
        }
        logger.debug(
            "IoTCounterProtocol is configured to use outer sensor ID: " +
            outer_sensor_id);

        if (props.containsKey(LED0_LINE)) {
            led0_line = Short.parseShort(props.getProperty(LED0_LINE));
            logger.debug("IoTCounterProtocol is configured to use led0 "
                         + "on digital line: " + led0_line);
        } else {
            logger.error("The applicaiton requires the line number of "
                         + "led1. Parameter: '" + LED0_LINE + "'");
            System.exit(1);
        }
        if (props.containsKey(LED1_LINE)) {
            led1_line = Short.parseShort(props.getProperty(LED1_LINE));
            logger.debug("IoTCounterProtocol is configured to use led1 "
                         + "on digital line: " + led1_line);
        } else {
            logger.error("The applicaiton requires the line number of "
                         + "led1. Parameter: '" + LED1_LINE + "'");
            System.exit(1);
        }

        // this.led0 = new DigitalOutputDevice(request, "led0", led0_line);
        // this.led1 = new DigitalOutputDevice(request, "led1", led1_line);

        this.inner_sensor = new DigitalInputDevice(
            "line_finder_in", inner_sensor_line, inner_sensor_id);
        inner_sensor.requestEventHandle(device,
                                        GpioEdgeDetection.FALLING);

        this.outer_sensor = new DigitalInputDevice(
            "line_finder_in", outer_sensor_line, outer_sensor_id);
        outer_sensor.requestEventHandle(device, GpioEdgeDetection.FALLING);

        Thread counter_thread = new Thread(this::counter);
        this.thread_id = counter_thread.threadId();
        counter_thread.start();
    }

    private void counter() {
        long inner_ts = 0, outer_ts = 0;
        String alias = myself.getAddress().getHostName();

        try {
            this.event = new GpioEvent();

            // what do we prefer?
            watcher.addHandle(inner_sensor.getEventHandle(),
                              inner_sensor.getID());
            outer_sensor.addHandle(watcher);

            byte[] buf;
            int event_id;
            while (true) {
                watcher.waitForEvent(event); // locks until event
                event_id = event.getId();

                if (event_id == inner_sensor.getLineNumber()) {
                    inner_ts = event.getNanoTimestamp();

                } else if (event_id == outer_sensor.getLineNumber()) {
                    outer_ts = event.getNanoTimestamp();

                } else {
                    continue;
                }

                if (inner_ts != 0 && outer_ts != 0) {
                    if (outer_ts - inner_ts > 0) {
                        buf = new UserMessage(myself.toString(), alias, "down")
                                  .toByteArray();
                    } else {
                        buf = new UserMessage(myself.toString(), alias, "up")
                                  .toByteArray();
                    }
                    inner_ts = 0;
                    outer_ts = 0;

                    sendRequest(new BroadcastRequest(
                                    myself, buf, IoTCounterProtocol.PROTO_ID),
                                (short)1600);

                    // lightUp();
                }
            }
        } catch (Exception e) {
            logger.error("Exception caught in counter: " + e.getMessage());
        }
    }

    private void lightUp() throws Exception {
        GpioLine led0_line = led0.getLine();
        GpioLine led1_line = led1.getLine();

        GpioBuffer buffer = new GpioBuffer();
        try (GpioHandle handle = device.requestHandle(request)) {
            nextLed(handle, buffer, led1_line, led0_line);
            nextLed(handle, buffer, led0_line, led1_line);
        } catch (Throwable e) {
            System.out.println("\nDamn, something went wrong!\n");
            e.printStackTrace();
        }
    }

    private static void nextLed(GpioHandle handle, GpioBuffer buffer,
                                GpioLine previousLed, GpioLine nextLed)
        throws InterruptedException, IOException {

        buffer.set(previousLed, false);

        buffer.set(nextLed, true);

        handle.write(buffer);

        Thread.sleep(500);
    }

    public long getThreadID() { return this.thread_id; }
}
