package pt.unl.fct.di.tardis.babel.iot.controlprotocols;

import java.io.IOException;
import java.util.Properties;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.pi4j.Pi4J;
import com.pi4j.context.Context;
import com.pi4j.exception.Pi4JException;
import com.pi4j.library.pigpio.PiGpio;
import com.pi4j.plugin.linuxfs.provider.i2c.LinuxFsI2CProvider;
import com.pi4j.plugin.pigpio.provider.gpio.digital.PiGpioDigitalInputProvider;
import com.pi4j.plugin.pigpio.provider.gpio.digital.PiGpioDigitalOutputProvider;
import com.pi4j.plugin.pigpio.provider.pwm.PiGpioPwmProvider;
import com.pi4j.plugin.pigpio.provider.serial.PiGpioSerialProvider;
import com.pi4j.plugin.pigpio.provider.spi.PiGpioSpiProvider;
import com.pi4j.plugin.raspberrypi.platform.RaspberryPiPlatform;

import pt.unl.fct.di.novasys.babel.core.GenericProtocol;
import pt.unl.fct.di.novasys.babel.exceptions.HandlerRegistrationException;
import pt.unl.fct.di.novasys.babel.protocols.dissemination.notifications.IdentifiableMessageNotification;
import pt.unl.fct.di.novasys.babel.protocols.general.notifications.ChannelAvailableNotification;
import pt.unl.fct.di.novasys.iot.device.i2c.GroveGestureDetector;
import pt.unl.fct.di.novasys.iot.device.i2c.GroveLcd;
import pt.unl.fct.di.novasys.iot.device.i2c.GroveLedMatrix;
import pt.unl.fct.di.tardis.babel.iot.controlprotocols.requests.SetDisplayColorRequest;
import pt.unl.fct.di.tardis.babel.iot.controlprotocols.requests.ShowAnimationRequest;
import pt.unl.fct.di.tardis.babel.iot.controlprotocols.requests.ShowEmojiRequest;
import pt.unl.fct.di.tardis.babel.iot.controlprotocols.requests.ShowGestureRequest;
import pt.unl.fct.di.tardis.babel.iot.controlprotocols.requests.ShowTextRequest;
import pt.unl.fct.di.tardis.babel.iot.controlprotocols.utils.I2CScanner;

public class IoTControlProtocolV2 extends GenericProtocol {

    private static final Logger logger =
        LogManager.getLogger(IoTControlProtocolV2.class);

    public final static short PROTO_ID = 8000;
    public final static String PROTO_NAME = "IoTControlProtocolV2";
    public final static String IOT_POLLING_RATE = "iot.polling.rate";
    public final static String IOT_DEFAULT_POLLING_RATE =
        "1000"; // milliseconds

    private final Context pi4j;
    private GroveLedMatrix matrix;
    private GroveLcd lcd;

    private byte[] lastState;

    private byte colors[] = {
        GroveLedMatrix.green,  GroveLedMatrix.red,    GroveLedMatrix.cyan,
        GroveLedMatrix.purple, GroveLedMatrix.yellow, GroveLedMatrix.pink,
        GroveLedMatrix.blue,   GroveLedMatrix.orange, GroveLedMatrix.white};

    private int currentX;
    private int currentY;
    private int maxCoordinate;

    private long thread_id;
    private int polling_rate;

    private I2CScanner scanner;

    public IoTControlProtocolV2() throws HandlerRegistrationException {
        super(IoTControlProtocolV2.PROTO_NAME, IoTControlProtocolV2.PROTO_ID);

        final PiGpio piGpio = PiGpio.newNativeInstance();

        pi4j = Pi4J.newContextBuilder()
                   .noAutoDetect()
                   .add(new RaspberryPiPlatform() {
                       @Override
                       protected String[] getProviders() {
                           return new String[] {};
                       }
                   })
                   .add(PiGpioDigitalInputProvider.newInstance(piGpio),
                        PiGpioDigitalOutputProvider.newInstance(piGpio),
                        PiGpioPwmProvider.newInstance(piGpio),
                        PiGpioSerialProvider.newInstance(piGpio),
                        PiGpioSpiProvider.newInstance(piGpio),
                        LinuxFsI2CProvider.newInstance())
                   .build();

        this.scanner = I2CScanner.getInstance();

        try {
            this.matrix = new GroveLedMatrix(pi4j);
        } catch (IOException e) {
            logger.error("Could not initialize the LedMatrix: ", e);
            this.matrix = null;
        } catch (Pi4JException pe) { // TODO better exception handling
            logger.error("Could not initialize the LedMatrix: ", pe);
            this.matrix = null;
        }

        try {
            this.lcd = new GroveLcd(pi4j);
        } catch (IOException e) {
            logger.error("Could not initialize the LCD: ", e);
            this.lcd = null;
        } catch (Pi4JException pe) { // TODO better exception handling
            logger.error("Could not initialize the LCD: ", pe);
            this.lcd = null;
        }

        if (this.matrix != null) {
            this.matrix.clearDisplay();

            this.lastState = this.matrix.getSnapshot();

            this.matrix.setAllColor((byte)255, (byte)0, (byte)0);

            registerMatrixMethods();
        }

        if (this.lcd != null) {
            registerLCDMethods();
        }

        this.currentX = 0;
        this.currentY = 0;
        this.maxCoordinate = 8;

        this.thread_id = 0L;
    }

    @Override
    public void init(Properties props)
        throws HandlerRegistrationException, IOException {
        String polling_rate_str =
            props.getProperty(IOT_POLLING_RATE, IOT_DEFAULT_POLLING_RATE);
        this.polling_rate = Integer.parseInt(polling_rate_str);

        Thread listener_thread = new Thread(this::deviceListener);
        this.thread_id = listener_thread.threadId();
        listener_thread.start();
    }

    private void deviceListener() {
        try {
            while (true) {
                scanner.detectDeviceActivity();

                Thread.sleep(this.polling_rate);
            }
        } catch (Exception e) {
            logger.error(e.getMessage());
        }
    }

    // TODO err on thing not available
    private void registerMatrixMethods() throws HandlerRegistrationException {
        registerRequestHandler(ShowAnimationRequest.REQUEST_ID,
                               this::handleShowAnimationRequest);
        registerRequestHandler(ShowGestureRequest.REQUEST_ID,
                               this::handleShowGestureRequest);
        registerRequestHandler(ShowEmojiRequest.REQUEST_ID,
                               this::handleShowEmojiRequest);
        registerRequestHandler(SetDisplayColorRequest.REQUEST_ID,
                               this::handleSetDisplayRequest);

        subscribeNotification(ChannelAvailableNotification.NOTIFICATION_ID,
                              this::handleChannelAvailableNotificaiton);
        subscribeNotification(IdentifiableMessageNotification.NOTIFICATION_ID,
                              this::handleIdentifiableMessageNotification);
    }

    private void registerLCDMethods() throws HandlerRegistrationException {
        registerRequestHandler(ShowTextRequest.REQUEST_ID,
                               this::handleShowTextRequest);
    }

    private void handleShowAnimationRequest(ShowAnimationRequest req,
                                            short protoId) {
        this.matrix.displayColorAnimation(req.getAnimation());
    }

    private void handleShowEmojiRequest(ShowEmojiRequest req, short protoId) {
        this.matrix.displayEmoji(req.getEmoji());
    }

    private void handleShowGestureRequest(ShowGestureRequest req,
                                          short protoId) {
        var gesture = req.getGesture();
        switch (gesture) {
        case GroveGestureDetector.PAJ7620GestureType.UP:
            System.out.println("up");
            this.matrix.setAllColor((byte)255, (byte)0, (byte)0);
            break;
        case GroveGestureDetector.PAJ7620GestureType.DOWN:
            System.out.println("down");
            this.matrix.setAllColor((byte)0, (byte)255, (byte)0);
            break;
        case GroveGestureDetector.PAJ7620GestureType.LEFT:
            System.out.println("left");
            this.matrix.setAllColor((byte)0, (byte)0, (byte)255);
            break;
        case GroveGestureDetector.PAJ7620GestureType.RIGHT:
            System.out.println("right");
            this.matrix.setAllColor((byte)255, (byte)0, (byte)255);
            break;
        case GroveGestureDetector.PAJ7620GestureType.PUSH:
            System.out.println("push");
            this.matrix.setAllColor((byte)255, (byte)255, (byte)0);
            break;
        case GroveGestureDetector.PAJ7620GestureType.PULL:
            System.out.println("pull");
            this.matrix.setAllColor((byte)0, (byte)255, (byte)255);
            break;
        default:
            break;
        }
    }

    private void handleShowTextRequest(ShowTextRequest req, short protoId) {
        this.lcd.setText(req.getText());
    }

    private void
    handleChannelAvailableNotificaiton(ChannelAvailableNotification not,
                                       short protoId) {
        this.matrix.setAllColor((byte)0, (byte)255, (byte)0);
    }

    private void handleSetDisplayRequest(SetDisplayColorRequest req,
                                         short protoId) {
        this.matrix.setAllColor((byte)req.getRed(), (byte)req.getGreen(),
                                (byte)req.getBlue());
    }

    private void
    handleIdentifiableMessageNotification(IdentifiableMessageNotification not,
                                          short portoId) {

        // Pick a color from the array that depends on the message identifier in
        // a deterministic but random way
        int color = Math.abs(
            ((int)(not.getMessage().getMID().getMostSignificantBits() % 16000) +
             (int)(not.getMessage().getMID().getLeastSignificantBits() %
                   16000)) %
            this.colors.length);

        this.matrix.loadSnapshot(lastState);
        try {
            this.matrix.setPixelColor(currentX, currentY, this.colors[color]);
        } catch (Exception e) {
            e.printStackTrace();
        }

        this.currentY++;
        if (this.currentY == this.maxCoordinate) {
            this.currentY = 0;
            this.currentX++;
            if (this.currentX == this.maxCoordinate)
                this.currentX = 0;
        }

        this.lastState = this.matrix.getSnapshot();
    }
}
