package pt.unl.fct.di.tardis.babel.iot.controlprotocols.utils;

import com.github.yafna.raspberry.grovepi.pi4j.GrovePi4J;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import pt.unl.fct.di.novasys.iot.device.i2c.Grove3AxisAccelerometer;
import pt.unl.fct.di.novasys.iot.device.i2c.GroveGestureDetector;
import pt.unl.fct.di.novasys.iot.device.i2c.GroveLcd;
import pt.unl.fct.di.novasys.iot.device.i2c.GroveLedMatrix;
import pt.unl.fct.di.tardis.babel.iot.api.DeviceType;

public class I2CScanner {
    private static final Logger logger = LogManager.getLogger(I2CScanner.class);

    private static I2CScanner instance;
    Set<DeviceType> connectedDevices;

    private I2CScanner() { this.connectedDevices = new HashSet<>(); }

    public static I2CScanner getInstance() {
        if (instance == null) {
            instance = new I2CScanner();
        }

        return instance;
    }

    public static Set<DeviceType> scan() {
        Set<DeviceType> devices = new HashSet<>();
        try {
            // could not find a programatic way to "ping" devices so...
            // immediate scan (-y) with i2cdetect
            Process proc = new ProcessBuilder("i2cdetect", "-y",
                                              String.valueOf(GrovePi4J.I2C_BUS))
                               .start();
            InputStreamReader isr =
                new InputStreamReader(proc.getInputStream());
            BufferedReader r = new BufferedReader(isr);

            String line;
            while ((line = r.readLine()) != null) {
                String[] parts;
                if ((parts = line.split(":")).length > 1) {
                    String[] addrs = parts[1].split("\\s+");
                    for (String addr : addrs) {
                        if (addr.length() == 2 && !addr.equals("--")) {
                            int device_addr = Integer.parseInt(addr, 16);
                            devices.add(getI2CAddressDevice(device_addr));
                        }
                    }
                }
            }

            proc.waitFor();
        } catch (Exception e) {
            logger.error(e.getMessage());
        }

        return devices;
    }

    public void detectDeviceActivity(Set<DeviceType> inUse) {
        probeDevices(inUse, true);
    }

    public void detectDeviceActivity() { probeDevices(scan(), false); }

    private void probeDevices(Set<DeviceType> currentDevices, boolean inUse) {
        for (DeviceType device : currentDevices) {
            if (connectedDevices.add(device)) {
                String addr_hex = getDeviceI2CAddressHex(device);
                logger.info("New device added at I2C address: " + addr_hex);
            }
        }

        for (DeviceType device : connectedDevices) {
            if (!currentDevices.contains(device)) {
                String addr_hex = getDeviceI2CAddressHex(device);
                if (inUse) {
                    logger.info("Disconnected device at I2C address: " +
                                addr_hex);

                } else {
                    logger.info("Disconnected device at I2C address: " +
                                addr_hex);
                }
                connectedDevices.remove(device);
            }
        }
    }

    public Set<DeviceType> getConnectedDevices() { return connectedDevices; }

    public List<Integer> getConnectedDeviceI2CAddresses() {
        return connectedDevices.stream()
            .map(d -> getDeviceI2CAddress(d))
            .filter(Objects::nonNull)
            .toList();
    }

    public static Integer getDeviceI2CAddress(DeviceType type) {
        switch (type) {
        case GROVE_LED_MATRIX:
            return GroveLedMatrix.LED_DISPLAY_ADDR;
        case GROVE_LCD:
            return GroveLcd.DISPLAY_TEXT_ADDR;
        case GROVE_GESTURE_DETECTOR:
            return GroveGestureDetector.PAJ7620_ADDR;
        case GROVE_3AXIS_ACCELERATOR:
            return Grove3AxisAccelerometer.MMA7660_ADDR;
        default:
            return null;
        }
    }

    public static DeviceType getI2CAddressDevice(int addr) {
        switch (addr) {
        case GroveLedMatrix.LED_DISPLAY_ADDR:
            return DeviceType.GROVE_LED_MATRIX;
        case GroveLcd.DISPLAY_TEXT_ADDR:
            return DeviceType.GROVE_LCD;
        case GroveGestureDetector.PAJ7620_ADDR:
            return DeviceType.GROVE_GESTURE_DETECTOR;
        case Grove3AxisAccelerometer.MMA7660_ADDR:
            return DeviceType.GROVE_3AXIS_ACCELERATOR;
        default:
            return null;
        }
    }

    public static String getDeviceI2CAddressHex(DeviceType type) {
        switch (type) {
        case GROVE_LED_MATRIX:
            return Integer.toHexString(GroveLedMatrix.LED_DISPLAY_ADDR);
        case GROVE_LCD:
            return Integer.toHexString(GroveLcd.DISPLAY_TEXT_ADDR);
        case GROVE_GESTURE_DETECTOR:
            return Integer.toHexString(GroveGestureDetector.PAJ7620_ADDR);
        case GROVE_3AXIS_ACCELERATOR:
            return Integer.toHexString(Grove3AxisAccelerometer.MMA7660_ADDR);
        default:
            return null;
        }
    }
}
