import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Scanner;
import java.util.TreeMap;
import java.util.TreeSet;

public class LogProcessor {

	private final Scanner sc;
	private final HashMap<String, MessageRecord> msgInfo;
	private final HashMap<String, List<String>> receivedMessagesPerNode;
	private final TreeSet<String> activeNodes;

	private final HashMap<String, String> mappingMessageToTopic;
	private final HashMap<String, List<String>> msgPerTopic;
	private final HashMap<String, TreeSet<String>> activeNodesPerTopic;

	private long firstMessageTransmitted;
	private HashMap<String, Long> firstMessageTransmittedInTopic;

	public LogProcessor(File input) throws FileNotFoundException {
		this.sc = new Scanner(input);
		this.msgInfo = new HashMap<String, MessageRecord>();
		this.receivedMessagesPerNode = new HashMap<String, List<String>>();
		this.activeNodes = new TreeSet<String>();

		this.mappingMessageToTopic = new HashMap<String, String>();
		this.msgPerTopic = new HashMap<String, List<String>>();
		this.activeNodesPerTopic = new HashMap<String, TreeSet<String>>();

		this.firstMessageTransmitted = Long.MAX_VALUE;
		this.firstMessageTransmittedInTopic = new HashMap<String, Long>();
	}

	public void readLog() {
		int l = 1;
		try {
			while (sc.hasNextLine()) {
				String[] tokens = sc.nextLine().split(" ");

				long timestamp = Long.parseLong(tokens[0]);
				String node = tokens[1];
				boolean create = tokens[2].equalsIgnoreCase("CRT");
				String msgId = tokens[3];
				String topic = tokens[5];
                this.firstMessageTransmittedInTopic.putIfAbsent(topic, Long.MAX_VALUE);

				if (create) {
					
					/* if(msgId.contains("|"))
						timestamp = Long.parseLong(msgId.split("\\|")[1]); */
					
					this.mappingMessageToTopic.put(msgId, topic);
					List<String> messages = this.msgPerTopic.get(topic);
					if (messages == null) {
						messages = new ArrayList<String>();
						this.msgPerTopic.put(topic, messages);
					}
					messages.add(msgId);

					if (timestamp < this.firstMessageTransmitted)
						this.firstMessageTransmitted = timestamp;

					if (timestamp < this.firstMessageTransmittedInTopic.get(topic))
						this.firstMessageTransmittedInTopic.put(topic, timestamp);
				}

				TreeSet<String> nodes = this.activeNodesPerTopic.get(topic);
				if (nodes == null) {
					nodes = new TreeSet<String>();
					this.activeNodesPerTopic.put(topic, nodes);
				}
				nodes.add(node);

				this.activeNodes.add(node);

				List<String> msgs = this.receivedMessagesPerNode.get(node);
				if (msgs == null) {
					msgs = new ArrayList<String>();
					this.receivedMessagesPerNode.put(node, msgs);
				}
				msgs.add(msgId);

				MessageRecord rec = this.msgInfo.get(msgId);

				if (rec == null) {
					if (create) {
						rec = new MessageRecord(msgId, node, timestamp);
					} else {
						rec = new MessageRecord(msgId);
						rec.registerMessageDelivery(node, timestamp);
					}
					this.msgInfo.put(msgId, rec);
				} else {
					if (create) {
						rec.setSenderNode(node);
						rec.setSentTImestamp(timestamp);
					} else {
						rec.registerMessageDelivery(node, timestamp);
					}
				}

				l++;
			}
		} catch (Exception e) {
			System.err.println("Error processing line " + l + " of the input file.");
			e.printStackTrace();
			System.exit(1);
		}
	}

	public void generateResults() throws FileNotFoundException {
		if (this.msgPerTopic.keySet().size() == 1) {
			// There is only a topic - compute only global aggregates
			this.computeReliability();
			this.computeLatency();
			this.computeCross();
		} else {
            for (var topicMsgs : this.msgPerTopic.entrySet()) {
                String topic = topicMsgs.getKey();
                File dir = new File(topic);
                dir.mkdir();

                this.computeReliability(topic, topicMsgs.getValue());
                this.computeLatency(topic, topicMsgs.getValue());
                this.computeCross(topic, topicMsgs.getValue());
            }
			// TODO: Fix this
		}
	}

	private void computeReliability(String topic, List<String> msgs) throws FileNotFoundException {
		TreeMap<Float, Integer> distribution = new TreeMap<Float, Integer>();
		TreeMap<Long, ArrayList<Float>> reliabilityOverTime = new TreeMap<Long, ArrayList<Float>>();

		long numberOfMessages = 0;

		List<String> duplicatedSends = new ArrayList<String>();
		
        MessageRecord rec;
		for (String msg : msgs) {
            rec = this.msgInfo.get(msg);

			numberOfMessages++;
			float reliability = (((float) (rec.getReceivedNodes().size() + 1)) / this.activeNodesPerTopic.get(topic).size()) * 100;
			
			if(reliability > 100) {
				System.err.println("Message " + rec.getMsgID() + " presents reliability of " + reliability + " (delivered/tx: " +
						(rec.getReceivedNodes().size()+1) + "; total nodes: " + this.activeNodesPerTopic.get(topic).size());
			
				duplicatedSends.add(rec.getMsgID());
				numberOfMessages--;
			}
			
			rec.setReliability(reliability);
			
			if (distribution.containsKey(reliability)) {
				distribution.put(reliability, distribution.get(reliability) + 1);
			} else {
				distribution.put(reliability, 1);
			}

			long timeline = rec.getSentTImestamp() - this.firstMessageTransmittedInTopic.get(topic);
			ArrayList<Float> reliabilities = reliabilityOverTime.get(timeline);
			if (reliabilities == null) {
				reliabilities = new ArrayList<Float>();
				reliabilityOverTime.put(timeline, reliabilities);
			}
			reliabilities.add(reliability);
		}

		for(String msg: duplicatedSends) {
			this.msgInfo.remove(msg);
		}

		// Generate output
		// Reliability Distribution && Reliability CDF
		PrintStream psD = new PrintStream(new File(topic + "/ReliabilityDistribution.txt"));
		PrintStream psC = new PrintStream(new File(topic + "/ReliabilityCDF.txt"));

		int accumulator = 0;

		for (Float s : distribution.keySet()) {
			int count = distribution.get(s);
			accumulator += count;

			psD.println(s + " " + (((float) count) / numberOfMessages) * 100);
			psC.println(s + " " + (((float) accumulator) / numberOfMessages) * 100);
		}

		psD.close();
		psC.close();

		// Reliability over time
		PrintStream psT = new PrintStream(new File(topic + "/ReliabilityOverTime.txt"));

		for (long timestamp : reliabilityOverTime.keySet()) {
			ArrayList<Float> rels = reliabilityOverTime.get(timestamp);
			float acc = 0;
			for (float f : rels)
				acc += f;
			acc = acc / rels.size();

			psT.println(timestamp + " " + acc);
		}

		psT.close();
	}

	private void computeReliability() throws FileNotFoundException {
		TreeMap<Float, Integer> distribution = new TreeMap<Float, Integer>();
		TreeMap<Long, ArrayList<Float>> reliabilityOverTime = new TreeMap<Long, ArrayList<Float>>();

		long numberOfMessages = 0;

		List<String> duplicatedSends = new ArrayList<String>();
		
		for (MessageRecord rec : this.msgInfo.values()) {

			numberOfMessages++;
			float reliability = (((float) (rec.getReceivedNodes().size() + 1)) / this.activeNodes.size()) * 100;
			
			if(reliability > 100) {
				System.err.println("Message " + rec.getMsgID() + " presents reliability of " + reliability + " (delivered/tx: " +
						(rec.getReceivedNodes().size()+1) + "; total nodes: " + this.activeNodes.size());			
			
				duplicatedSends.add(rec.getMsgID());
				numberOfMessages--;
			}
			
			rec.setReliability(reliability);
			
			if (distribution.containsKey(reliability)) {
				distribution.put(reliability, distribution.get(reliability) + 1);
			} else {
				distribution.put(reliability, 1);
			}

			long timeline = rec.getSentTImestamp() - this.firstMessageTransmitted;
			ArrayList<Float> reliabilities = reliabilityOverTime.get(timeline);
			if (reliabilities == null) {
				reliabilities = new ArrayList<Float>();
				reliabilityOverTime.put(timeline, reliabilities);
			}
			reliabilities.add(reliability);
		}

		for(String msg: duplicatedSends) {
			this.msgInfo.remove(msg);
		}
		
		// Generate output
		// Reliability Distribution && Reliability CDF
		PrintStream psD = new PrintStream(new File("ReliabilityDistribution.txt"));
		PrintStream psC = new PrintStream(new File("ReliabilityCDF.txt"));

		int accumulator = 0;

		for (Float s : distribution.keySet()) {
			int count = distribution.get(s);
			accumulator += count;

			psD.println(s + " " + (((float) count) / numberOfMessages) * 100);
			psC.println(s + " " + (((float) accumulator) / numberOfMessages) * 100);
		}

		psD.close();
		psC.close();

		// Reliability over time
		PrintStream psT = new PrintStream(new File("ReliabilityOverTime.txt"));

		for (long timestamp : reliabilityOverTime.keySet()) {
			ArrayList<Float> rels = reliabilityOverTime.get(timestamp);
			float acc = 0;
			for (float f : rels)
				acc += f;
			acc = acc / rels.size();

			psT.println(timestamp + " " + acc);
		}

		psT.close();
	}

	private void computeLatency(String topic, List<String> msgs) throws FileNotFoundException {
		TreeMap<Long, Integer> distribution = new TreeMap<Long, Integer>();
		TreeMap<Long, ArrayList<Long>> latencyOverTime = new TreeMap<Long, ArrayList<Long>>();

		long numberOfMessages = 0;

        MessageRecord rec;
		for (String msg : msgs) {
            rec = this.msgInfo.get(msg);

			numberOfMessages++;
			long latency = (rec.getReceivedTimestamps().size() > 0 ? rec.getReceivedTimestamps().getLast() - rec.getSentTImestamp() : 0);
			rec.setLatency(latency);
			
			if (distribution.containsKey(latency)) {
				distribution.put(latency, distribution.get(latency) + 1);
			} else {
				distribution.put(latency, 1);
			}

			long timeline = rec.getSentTImestamp() - this.firstMessageTransmittedInTopic.get(topic);
			ArrayList<Long> latencies = latencyOverTime.get(timeline);
			if (latencies == null) {
				latencies = new ArrayList<Long>();
				latencyOverTime.put(timeline, latencies);
			}
			latencies.add(latency);
		}

		// Generate output
		// Reliability Distribution && Reliability CDF
		PrintStream psD = new PrintStream(new File(topic + "/LatencyDistribution.txt"));
		PrintStream psC = new PrintStream(new File(topic + "/LatencyCDF.txt"));

		int accumulator = 0;

		for (Long s : distribution.keySet()) {
			int count = distribution.get(s);
			accumulator += count;

			psD.println(s + " " + (((float) count) / numberOfMessages) * 100);
			psC.println(s + " " + (((float) accumulator) / numberOfMessages) * 100);
		}

		psD.close();
		psC.close();

		// Reliability over time
		PrintStream psT = new PrintStream(new File(topic + "/LatencyOverTime.txt"));

		for (long timestamp : latencyOverTime.keySet()) {
			ArrayList<Long> rels = latencyOverTime.get(timestamp);
			float acc = 0;
			for (long l : rels)
				acc += l;
			acc = acc / rels.size();

			psT.println(timestamp + " " + acc);
		}

		psT.close();
	}

	private void computeLatency() throws FileNotFoundException {
		TreeMap<Long, Integer> distribution = new TreeMap<Long, Integer>();
		TreeMap<Long, ArrayList<Long>> latencyOverTime = new TreeMap<Long, ArrayList<Long>>();

		long numberOfMessages = 0;

		for (MessageRecord rec : this.msgInfo.values()) {

			numberOfMessages++;
			long latency = (rec.getReceivedTimestamps().size() > 0 ? rec.getReceivedTimestamps().getLast() - rec.getSentTImestamp() : 0);
			rec.setLatency(latency);
			
			if (distribution.containsKey(latency)) {
				distribution.put(latency, distribution.get(latency) + 1);
			} else {
				distribution.put(latency, 1);
			}

			long timeline = rec.getSentTImestamp() - this.firstMessageTransmitted;
			ArrayList<Long> latencies = latencyOverTime.get(timeline);
			if (latencies == null) {
				latencies = new ArrayList<Long>();
				latencyOverTime.put(timeline, latencies);
			}
			latencies.add(latency);
		}

		// Generate output
		// Reliability Distribution && Reliability CDF
		PrintStream psD = new PrintStream(new File("LatencyDistribution.txt"));
		PrintStream psC = new PrintStream(new File("LatencyCDF.txt"));

		int accumulator = 0;

		for (Long s : distribution.keySet()) {
			int count = distribution.get(s);
			accumulator += count;

			psD.println(s + " " + (((float) count) / numberOfMessages) * 100);
			psC.println(s + " " + (((float) accumulator) / numberOfMessages) * 100);
		}

		psD.close();
		psC.close();

		// Reliability over time
		PrintStream psT = new PrintStream(new File("LatencyOverTime.txt"));

		for (long timestamp : latencyOverTime.keySet()) {
			ArrayList<Long> rels = latencyOverTime.get(timestamp);
			float acc = 0;
			for (long l : rels)
				acc += l;
			acc = acc / rels.size();

			psT.println(timestamp + " " + acc);
		}

		psT.close();
	}

	private void computeCross(String topic, List<String> msgs) throws FileNotFoundException {
		// Crosscheck between latency and reliability
		// PrintStream psX = new PrintStream(new File("ReliabilityXLatency_" + topic + ".txt"));
		PrintStream psX = new PrintStream(new File(topic + "/ReliabilityXLatency.txt"));

        MessageRecord rec;
		for (String msg : msgs) {
            rec = this.msgInfo.get(msg);

			psX.println(rec.getReliability() + " " + rec.getLatency());
		}

		psX.close();
	}

	private void computeCross() throws FileNotFoundException {
		// Crosscheck between latency and reliability
		PrintStream psX = new PrintStream(new File("ReliabilityXLatency.txt"));

		for (MessageRecord rec : this.msgInfo.values()) {

			psX.println(rec.getReliability() + " " + rec.getLatency());
		}

		psX.close();
	}

	public static void main(String[] args) throws FileNotFoundException {
		if (args.length != 1) {
			System.err.println("Usage: java " + LogProcessor.class.getCanonicalName() + " <input file>");
			System.exit(1);
		}

		LogProcessor lp = new LogProcessor(new File(args[0]));

		lp.readLog();

		lp.generateResults();
	}

	@SuppressWarnings("unused")
	private class MessageRecord {

		private final String msgID;
		private long sentTImestamp;
		private List<Long> receivedTimestamps;
		private String senderNode;
		private List<String> receivedNodes;

		private long latency;
		private float reliability;

		public MessageRecord(String msgID) {
			this.msgID = msgID;
			this.sentTImestamp = -1;
			this.receivedTimestamps = new ArrayList<Long>();
			this.senderNode = null;
			this.receivedNodes = new ArrayList<String>();
		}

		public MessageRecord(String msgID, String sender, long creationTime) {
			this(msgID);
			this.setSenderNode(sender);
			this.setSentTImestamp(creationTime);
		}

		public long getSentTImestamp() {
			return sentTImestamp;
		}

		public void setSentTImestamp(long sentTImestamp) {
			this.sentTImestamp = sentTImestamp;
		}

		public List<Long> getReceivedTimestamps() {
			Collections.sort(receivedTimestamps);
			return receivedTimestamps;
		}

		public void addReceivedTimestamps(long timestamp) {
			this.receivedTimestamps.add(timestamp);
		}

		public String getSenderNode() {
			return senderNode;
		}

		public void setSenderNode(String senderNode) {
			this.senderNode = senderNode;
		}

		public List<String> getReceivedNodes() {
			return receivedNodes;
		}

		public void setReceivedNodes(String receiver) {
			this.receivedNodes.add(receiver);
		}

		public String getMsgID() {
			return msgID;
		}

		public void registerMessageDelivery(String node, long timestamp) {
			this.receivedNodes.add(node);
			this.receivedTimestamps.add(timestamp);
		}

		public long getLatency() {
			return latency;
		}

		public void setLatency(long latency) {
			this.latency = latency;
		}

		public float getReliability() {
			return reliability;
		}

		public void setReliability(float reliability) {
			this.reliability = reliability;
		}

        @Override
        public String toString() {
            return "MessageRecord{" +
					"msgID='" + msgID + ",\n" +
					"sentTImestamp=" + sentTImestamp + ",\n" +
					"receivedTimestamps=" + receivedTimestamps + ",\n" +
					"senderNode='" + senderNode + ",\n" +
					"receivedNodes=" + receivedNodes + ",\n" +
					"latency=" + latency + ",\n" +
					"reliability=" + reliability +
					'}';
        }
	}
}
